package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.Addr;
import biz.everit.audit.hook.constant.App;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.Address;
import com.liferay.portal.model.BaseModelListener;

/**
 * 
 * Address hook.
 * 
 */
public class AddressModificationAudit extends BaseModelListener<Address> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();
    /**
     * CREATE_ADDRESS.
     */
    private static final String CREATE_ADDRESS = "createAddress";
    /**
     * UPDATE_ADDRESS.
     */
    private static final String UPDATE_ADDRESS = "updateAddress";
    /**
     * REMOVE_ADDRESS.
     */
    private static final String REMOVE_ADDRESS = "removeAddress";

    /**
     * Converts the address to event.
     * 
     * @param eventName
     *            Name of address modification event (create/update/delete).
     * @param address
     *            The modified address object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromAddress(final String eventName,
            final Address address) {

        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Addr.ADDRESS_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(address.getAddressId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.USER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(address.getUserId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.STREET1);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(address.getStreet1());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.STREET2);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(address.getStreet2());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.STREET3);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(address.getStreet3());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.CITY);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(address.getCity());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.ZIP);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(address.getZip());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Addr.COUNTRY_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(address.getCountryId()));
        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of Address created for audit component.
     * 
     * @param address
     *            The address changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final Address address) throws ModelListenerException {
        Event event = getEventFromAddress(CREATE_ADDRESS, address);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Address removed for audit component.
     * 
     * @param address
     *            The address changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final Address address) throws ModelListenerException {
        Event event = getEventFromAddress(REMOVE_ADDRESS, address);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Address updated for audit component.
     * 
     * @param address
     *            The address changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final Address address) throws ModelListenerException {
        Event event = getEventFromAddress(UPDATE_ADDRESS, address);
        auditClient.logEvent(event);
    }

}
