package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Cont;
import biz.everit.audit.hook.util.DateUtil;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portal.model.Contact;

/**
 * 
 * Contact hook.
 * 
 */
public class ContactModificationAudit extends BaseModelListener<Contact> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_CONTACT.
     */
    private static final String CREATE_CONTACT = "createContact";
    /**
     * UPDATE_CONTACT.
     */
    private static final String UPDATE_CONTACT = "updateContact";
    /**
     * REMOVE_CONTACT.
     */
    private static final String REMOVE_CONTACT = "removeContact";

    /**
     * 
     * Converts the contact to event.
     * 
     * @param eventName
     *            Name of contact modification event (create/update/delete).
     * @param contact
     *            The modified contact object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromContact(final String eventName,
            final Contact contact) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Cont.CONTACT_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(contact.getContactId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Cont.USER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(contact.getUserId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Cont.MALE);
        ed.setEventDataType(EventDataType.NUMBER);
        if (contact.isMale()) {
            ed.setNumberValue(new Double(1));
        } else {
            ed.setNumberValue(new Double(0));
        }
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Cont.BIRTHDAY);
        ed.setEventDataType(EventDataType.TIMESTAMP);
        ed.setTimestampValue(DateUtil.dateToCal(contact.getBirthday()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Cont.JOBTITLE);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(contact.getJobTitle());
        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of Contact created for audit component.
     * 
     * @param contact
     *            The contact changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final Contact contact)
            throws ModelListenerException {
        Event event = getEventFromContact(CREATE_CONTACT, contact);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Contact removed for audit component.
     * 
     * @param contact
     *            The contact changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final Contact contact)
            throws ModelListenerException {
        Event event = getEventFromContact(REMOVE_CONTACT, contact);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Contact updated for audit component.
     * 
     * @param contact
     *            The contact changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final Contact contact)
            throws ModelListenerException {
        Event event = getEventFromContact(UPDATE_CONTACT, contact);
        auditClient.logEvent(event);
    }
}
