package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Exp;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portlet.expando.model.ExpandoValue;

/**
 * 
 * Expando hook.
 * 
 */
public class ExpandoValueModificationAudit extends BaseModelListener<ExpandoValue> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_EXPANDO.
     */
    private static final String CREATE_EXPANDO = "createExpando";
    /**
     * UPDATE_EXPANDO.
     */
    private static final String UPDATE_EXPANDO = "updateExpando";
    /**
     * REMOVE_EXPANDO.
     */
    private static final String REMOVE_EXPANDO = "removeExpando";

    /**
     * 
     * Converts the expando to event.
     * 
     * @param eventName
     *            Name of expando modification event (create/update/delete).
     * @param expando
     *            The modified expando object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromExpando(final String eventName,
            final ExpandoValue expando) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Exp.VALUE_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(expando.getValueId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Exp.TABLE_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(expando.getTableId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Exp.COLUMN_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(expando.getColumnId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Exp.ROW_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(expando.getRowId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Exp.CLASSNAME_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(expando.getClassNameId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Exp.DATA);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(expando.getData());
        eventDataArray.add(ed);
        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));

        return event;
    }

    /**
     * Sends the Event of Expando created for audit component.
     * 
     * @param expando
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final ExpandoValue expando)
            throws ModelListenerException {
        Event event = getEventFromExpando(CREATE_EXPANDO, expando);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Expando removed for audit component.
     * 
     * @param expando
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final ExpandoValue expando)
            throws ModelListenerException {
        Event event = getEventFromExpando(REMOVE_EXPANDO, expando);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Expando updated for audit component.
     * 
     * @param expando
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final ExpandoValue expando)
            throws ModelListenerException {
        Event event = getEventFromExpando(UPDATE_EXPANDO, expando);
        auditClient.logEvent(event);
    }

}
