package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.DLFE;
import biz.everit.audit.hook.util.DateUtil;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portlet.documentlibrary.model.DLFileEntry;

/**
 * 
 * DLFileEntry hook.
 * 
 */
public class FileEntryModificationAudit extends BaseModelListener<DLFileEntry> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_FILE.
     */
    private static final String CREATE_FILE = "createDLFileEntry";
    /**
     * UPDATE_FILE.
     */
    private static final String UPDATE_FILE = "updateDLFileEntry";
    /**
     * CREATE_FILE.
     */
    private static final String REMOVE_FILE = "removeDLFileEntry";

    /**
     * 
     * Converts the fileEntry to event.
     * 
     * @param eventName
     *            Name of file modification event (create/update/delete).
     * @param file
     *            The modified file object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromDLFileEntry(final String eventName,
            final DLFileEntry file) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(DLFE.CREATE_DATE);
        ed.setEventDataType(EventDataType.TIMESTAMP);
        ed.setTimestampValue(DateUtil.dateToCal(file.getCreateDate()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.DESCRIPTION);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(file.getDescription());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.EXTENSION);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(file.getExtension());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.FILE_ENTRY_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(file.getFileEntryId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.FOLDER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(file.getFolderId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.MODIFIED_DATE);
        ed.setEventDataType(EventDataType.TIMESTAMP);
        ed.setTimestampValue(DateUtil.dateToCal(file.getModifiedDate()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.NAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(file.getName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.SIZE);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(file.getSize()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.USER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(file.getUserId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.UU_ID);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(file.getUuid());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(DLFE.VERSION);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(file.getVersion());
        eventDataArray.add(ed);

        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of DLFileEntry created for audit component.
     * 
     * @param file
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final DLFileEntry file)
            throws ModelListenerException {
        Event event = getEventFromDLFileEntry(CREATE_FILE, file);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of DLFileEntry removed for audit component.
     * 
     * @param file
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final DLFileEntry file)
            throws ModelListenerException {
        Event event = getEventFromDLFileEntry(REMOVE_FILE, file);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of DLFileEntry updated for audit component.
     * 
     * @param file
     *            The expando changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final DLFileEntry file)
            throws ModelListenerException {
        Event event = getEventFromDLFileEntry(UPDATE_FILE, file);
        auditClient.logEvent(event);
    }

}
