package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Org;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portal.model.Organization;

/**
 * 
 * Organization hook.
 * 
 */
public class OrganizationModificationAudit extends BaseModelListener<Organization> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_ORGANIZATION.
     */
    private static final String CREATE_ORGANIZATION = "createOrganization";
    /**
     * UPDATE_ORGANIZATION.
     */
    private static final String UPDATE_ORGANIZATION = "updateOrganization";
    /**
     * REMOVE_ORGANIZATION.
     */
    private static final String REMOVE_ORGANIZATION = "removeOrganization";

    /**
     * 
     * Converts the organization to event.
     * 
     * @param eventName
     *            Name of organization modification event (create/update/delete).
     * @param organization
     *            The modified organization object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromOrganization(final String eventName,
            final Organization organization) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Org.COMPANY_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getCompanyId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.COUNTRY_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getCountryId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.NAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(organization.getName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.ORG_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getOrganizationId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.PARENT_ORG_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getParentOrganizationId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.REGION_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getRegionId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.STATUS_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(organization.getStatusId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Org.TYPE);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(organization.getType());
        eventDataArray.add(ed);

        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of Organization created for audit component.
     * 
     * @param organization
     *            The organization changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final Organization organization)
            throws ModelListenerException {
        Event event =
                getEventFromOrganization(CREATE_ORGANIZATION, organization);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Organization removed for audit component.
     * 
     * @param organization
     *            The organization changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final Organization organization)
            throws ModelListenerException {
        Event event =
                getEventFromOrganization(REMOVE_ORGANIZATION, organization);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Organization updated for audit component.
     * 
     * @param organization
     *            The organization changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final Organization organization)
            throws ModelListenerException {
        Event event =
                getEventFromOrganization(UPDATE_ORGANIZATION, organization);
        auditClient.logEvent(event);
    }

}
