package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Perm;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portal.model.ResourcePermission;

/**
 * 
 * ResourcePermission hook.
 * 
 */
public class PermissionModificationAudit extends BaseModelListener<ResourcePermission> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_PERMISSION.
     */
    private static final String CREATE_PERMISSION = "createPermission";
    /**
     * UPDATE_PERMISSION.
     */
    private static final String UPDATE_PERMISSION = "updatePermission";
    /**
     * REMOVE_PERMISSION.
     */
    private static final String REMOVE_PERMISSION = "removePermission";

    /**
     * Converts the resourcePermission to event.
     * 
     * @param eventName
     *            Name of permission modification event (create/update/delete).
     * @param resperm
     *            The modified permission object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFrompermission(final String eventName,
            final ResourcePermission resperm) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Perm.NAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue((resperm.getName()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Perm.PRIM_KEY);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(resperm.getPrimKey());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Perm.COMPANY_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(Double.valueOf(resperm.getCompanyId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Perm.PRIMARY_KEY);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(Double.valueOf(resperm.getPrimaryKey()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setName(Perm.SCOPE);
        ed.setNumberValue(Double.valueOf(resperm.getScope()));
        eventDataArray.add(ed);
        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of permission created for audit component.
     * 
     * @param resperm
     *            The resperm changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final ResourcePermission resperm)
            throws ModelListenerException {
        Event event = getEventFrompermission(CREATE_PERMISSION, resperm);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of permission removed for audit component.
     * 
     * @param resperm
     *            The resperm changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final ResourcePermission resperm)
            throws ModelListenerException {
        Event event = getEventFrompermission(REMOVE_PERMISSION, resperm);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of permission updated for audit component.
     * 
     * @param resperm
     *            The resperm changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final ResourcePermission resperm)
            throws ModelListenerException {
        Event event = getEventFrompermission(UPDATE_PERMISSION, resperm);
        auditClient.logEvent(event);
    }

}
