package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Pho;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portal.model.Phone;

/**
 * 
 * Phone hook.
 * 
 */
public class PhoneModificationAudit extends BaseModelListener<Phone> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_PHONE.
     */
    private static final String CREATE_PHONE = "createPhone";
    /**
     * UPDATE_PHONE.
     */
    private static final String UPDATE_PHONE = "updatePhone";
    /**
     * REMOVE_PHONE.
     */
    private static final String REMOVE_PHONE = "removePhone";

    /**
     * Converts the address to event.
     * 
     * @param eventName
     *            Name of phone modification event (create/update/delete).
     * @param phone
     *            The modified phone object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromPhone(final String eventName, final Phone phone) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Pho.PHONE_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(phone.getPhoneId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Pho.USER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(phone.getUserId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Pho.NUMBER);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(phone.getNumber());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Pho.TYPE_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(phone.getTypeId()));
        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of Phone created for audit component.
     * 
     * @param phone
     *            The phone changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final Phone phone) throws ModelListenerException {
        Event event = getEventFromPhone(CREATE_PHONE, phone);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Phone removed for audit component.
     * 
     * @param phone
     *            The phone changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final Phone phone) throws ModelListenerException {
        Event event = getEventFromPhone(REMOVE_PHONE, phone);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of Phone updated for audit component.
     * 
     * @param phone
     *            The phone changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final Phone phone) throws ModelListenerException {
        Event event = getEventFromPhone(UPDATE_PHONE, phone);
        auditClient.logEvent(event);
    }

}
