package biz.everit.audit.hook;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.List;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.hook.client.AuditClient;
import biz.everit.audit.hook.constant.App;
import biz.everit.audit.hook.constant.Us;

import com.liferay.portal.ModelListenerException;
import com.liferay.portal.model.BaseModelListener;
import com.liferay.portal.model.User;

/**
 * 
 * User hook.
 * 
 */
public class UserModificationAudit extends BaseModelListener<User> {
    /**
     * Audit client.
     */
    private final AuditClient auditClient = new AuditClient();

    /**
     * CREATE_USER.
     */
    private static final String CREATE_USER = "createUser";
    /**
     * UPDATE_USER.
     */
    private static final String UPDATE_USER = "updateUser";
    /**
     * REMOVE_USER.
     */
    private static final String REMOVE_USER = "removeUser";

    /**
     * 
     * Converts the user to event.
     * 
     * @param eventName
     *            Name of user modification event (create/update/delete).
     * @param user
     *            The modified user object.
     * @return With an Event object for audit web service.
     */
    private Event getEventFromUser(final String eventName, final User user) {
        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(Us.USER_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(user.getUserId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.SCREENNAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getScreenName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.EMAIL_ADDRESS);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getEmailAddress());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.FIRSTNAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getFirstName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.MIDDLENAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getMiddleName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.LASTNAME);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getLastName());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.JOBTITLE);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(user.getJobTitle());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.PORTRAIT_ID);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(new Double(user.getPortraitId()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(Us.ACTIVE);
        ed.setEventDataType(EventDataType.NUMBER);
        if (user.isActive()) {
            ed.setNumberValue(new Double(1));
        } else {
            ed.setNumberValue(new Double(0));
        }
        eventDataArray.add(ed);

        Event event = new Event(null,
                eventName, App.DEFAULT_APP_NAME, null, eventDataArray.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Sends the Event of User created for audit component.
     * 
     * @param user
     *            The user changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterCreate(final User user) throws ModelListenerException {
        Event event = getEventFromUser(CREATE_USER, user);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of User removed for audit component.
     * 
     * @param user
     *            The user changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterRemove(final User user) throws ModelListenerException {
        Event event = getEventFromUser(REMOVE_USER, user);
        auditClient.logEvent(event);
    }

    /**
     * Sends the Event of User updated for audit component.
     * 
     * @param user
     *            The user changed.
     * @throws ModelListenerException
     *             If an exception happened.
     */
    @Override
    public void onAfterUpdate(final User user) throws ModelListenerException {
        Event event = getEventFromUser(UPDATE_USER, user);
        auditClient.logEvent(event);
    }

}
