package biz.everit.audit.hook.client;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import javax.naming.NamingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.hook.util.EventFactory;
import biz.everit.audit.jms.client.service.AuditJmsService;
import biz.everit.audit.jms.client.service.impl.AuditJmsServiceImpl;

/**
 * 
 * Ws client for auditing.
 * 
 */
public class AuditClient {

    /**
     * Logger for this class.
     */
    protected static final Logger LOGGER = LoggerFactory.getLogger(AuditClient.class);
    /**
     * JMS connection factory name.
     */
    private static final String JMS_CONNECTION_FACTORY_NAME = "jms/auditJmsConnectionFactory";
    /**
     * JMS queue name.
     */
    private static final String JMS_QUEUE_NAME = "jms/auditJmsQueue";
    /**
     * The JMS service.
     */
    private AuditJmsService jmsService;

    /**
     * Constructor for initializing the class, and creating the JMS service.
     */
    public AuditClient() {
        try {
            jmsService = new AuditJmsServiceImpl(JMS_CONNECTION_FACTORY_NAME, JMS_QUEUE_NAME);
        } catch (NamingException e) {
            LOGGER.error("Naming exception: ", e);
        }
    }

    /**
     * Constructs string representation of the eventDataArray.
     * 
     * @param eventDataArray
     *            - eventDataArray
     * @return the string
     */
    private String eventDatasToString(final EventData[]
            eventDataArray) {
        final StringBuilder sb = new StringBuilder();
        sb.append("[ ");
        int i = 0;
        for (final EventData eventData : eventDataArray) {
            sb.append(" [" + i + ": " + eventData.getName());
            if (eventData.getBinaryValue() != null) {
                sb.append(", binary value: "
                        + eventData.getBinaryValue().length + " bytes long");
            }
            if (eventData.getNumberValue() != null) {
                sb.append(", number value: " + eventData.getNumberValue());
            }
            if (eventData.getTextValue() != null) {
                sb.append(", text value: " + eventData.getTextValue());
            }
            if (eventData.getTimestampValue() != null) {
                sb
                        .append(", timestamp value: "
                                + eventData.getTimestampValue());
            }
            sb.append(" ]");
            i++;
        }
        sb.append(" ]");
        return sb.toString();
    }

    /**
     * Log event.
     * 
     * @param event
     *            Event to be logged.
     */
    public void logEvent(final Event event) {
        Event filledEvent = EventFactory.fillEvent(event);
        if (jmsService != null) {
            jmsService.log(filledEvent);
        } else {
            LOGGER.error("Cannot audit, because audit client is null!");
            LOGGER
                    .debug("Appname: " + event.getApplicationName() + " EventType: "
                            + event.getName() + " Data: "
                            + eventDatasToString(event.getEventDataArray()));
        }
    }
}
