package biz.everit.audit.hook.util;

/*
 * Copyright (c) 2011, Everit Kft.
 *
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.transaction.TransactionSynchronizationRegistry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import biz.everit.audit.api.dto.Event;
import biz.everit.audit.api.dto.EventData;
import biz.everit.audit.api.dto.EventDataType;
import biz.everit.audit.util.context.ServletRequestContext;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.model.User;
import com.liferay.portal.security.permission.PermissionChecker;
import com.liferay.portal.security.permission.PermissionThreadLocal;
import com.liferay.portal.service.UserLocalServiceUtil;

/**
 * 
 * Factory class for prefilled event generation.
 * 
 */
public final class EventFactory {
    /**
     * SESSION_TYPE.
     */
    private static final String ACTOR = User.class.getName();
    /**
     * SESSION_TYPE.
     */
    private static final String SESSION_TYPE = "session";
    /**
     * TRANSACTION_TYPE.
     */
    private static final String TRANSACTION_TYPE = "transaction";
    /**
     * ACTOR_EMAIL.
     */
    private static final String ACTOR_EMAIL = "userEmail";
    /**
     * Logger for this utility class.
     */
    protected static final Logger LOGGER =
            LoggerFactory.getLogger(EventFactory.class);

    /**
     * Create event with prefilled values (userid, email, transactionId, sessionId).
     * 
     * @return The cretated event.
     */
    private static List<EventData> createPrefilledEventDatas() {

        List<EventData> eventDataArray = new ArrayList<EventData>();

        EventData ed = new EventData();
        ed.setName(EventFactory.SESSION_TYPE);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(getSessionId());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(EventFactory.TRANSACTION_TYPE);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(getTransactionId());
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(EventFactory.ACTOR);
        ed.setEventDataType(EventDataType.NUMBER);
        ed.setNumberValue(Double.valueOf(getUserId().doubleValue()));
        eventDataArray.add(ed);

        ed = new EventData();
        ed.setName(EventFactory.ACTOR_EMAIL);
        ed.setEventDataType(EventDataType.STRING);
        ed.setTextValue(getUserEmail());
        eventDataArray.add(ed);
        return eventDataArray;
    }

    /**
     * Fill event with default parameters (session, user, transaction).
     * 
     * @param event
     *            The event to which the default properties are added.
     * @return The filled event.
     */
    public static Event fillEvent(final Event event) {
        EventData[] eventDataArray = event.getEventDataArray();
        List<EventData> original = new ArrayList<EventData>(Arrays.asList(eventDataArray));
        original.addAll(createPrefilledEventDatas());
        event.setEventDataArray(original.toArray(new EventData[] {}));
        return event;
    }

    /**
     * Get the id of the audited session.
     * 
     * @return Id of the audited session.
     */
    private static String getSessionId() {
        ServletRequest request = ServletRequestContext.getServletRequest();
        if (request != null) {
            if (request instanceof HttpServletRequest) {
                HttpServletRequest servletRequest =
                        (HttpServletRequest) request;
                HttpSession session = servletRequest.getSession(false);
                if (session != null) {
                    return session.getId();
                }
            }
            LOGGER.debug("No session id is associated with this session");
        }
        return null;
    }

    /**
     * Get the id of the audited transaction.
     * 
     * @return Id of the audited transaction.
     */
    private static String getTransactionId() {
        Context ctx;
        try {
            ctx = new InitialContext();
            TransactionSynchronizationRegistry registry =
                    (TransactionSynchronizationRegistry) ctx
                            .lookup("java:comp/TransactionSynchronizationRegistry");
            if (registry != null) {
                Object transactionKey = registry.getTransactionKey();
                if (transactionKey != null) {
                    return transactionKey.toString();
                }
            }
        } catch (NamingException e) {
            LOGGER.error("javax.naming.NamingException");
        }
        LOGGER.debug("No transaction id is associated with this transaction"
                + ", defaulting to 0");
        return "0";
    }

    /**
     * Get the email of the audited user.
     * 
     * @return The email address of the audited user.
     */
    private static String getUserEmail() {
        Long userId = getUserId();
        if (userId.longValue() != 0) {
            try {
                return UserLocalServiceUtil.getUserById(userId)
                        .getEmailAddress();
            } catch (PortalException e) {
                LOGGER.debug("No user is associated with this transaction, "
                        + "defaulting to 0");
            } catch (SystemException e) {
                LOGGER.debug("No user is associated with this transaction, "
                        + "defaulting to 0");
            }
        }
        return null;
    }

    /**
     * Get the id of the audited user.
     * 
     * @return The id of the audited user.
     */
    private static Long getUserId() {
        final PermissionChecker permissionChecker =
                PermissionThreadLocal.getPermissionChecker();
        long userId = 0;
        if (permissionChecker == null) {
            LOGGER.debug("No user is associated with this transaction, "
                    + "defaulting to 0");
        } else {
            userId = permissionChecker.getUserId();
        }
        return userId;
    }

    /**
     * Private constructor for utility class.
     */
    private EventFactory() {
        super();
    }

}
