/**
 * 
 */
package hu.everit.appserver.utils;

import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

/**
 * Application server wide bean locator. It searches the contained bean locators
 * to search for beans with the given names throughout the bean contexts covered
 * by the bean locators.
 * 
 * @author Balazs Zsoldos, Sandor Nemeth
 * @since 1.0
 */
public class BeanLocatorUtil {

	/**
	 * Entry class.
	 * 
	 * @author sandor.nemeth
	 */
	private static class BeanLocatorEntry {
		/**
		 * The bean locator registered to this entry.
		 */
		public BeanLocator beanLocator;
		/**
		 * The beans loaded from this bean locator.
		 */
		public Set<String> containedBeans = new HashSet<String>();

		/**
		 * Constructor.
		 * 
		 * @param beanLocator
		 *            the bean locator
		 */
		public BeanLocatorEntry(BeanLocator beanLocator) {
			this.beanLocator = beanLocator;
		}
	}

	/**
	 * The registered bean locators.
	 */
	private static Map<String, BeanLocatorEntry> beanLocators = new HashMap<String, BeanLocatorEntry>();

	/**
	 * The beans already loaded.
	 */
	private static Map<String, Object> beans = new HashMap<String, Object>();

	/**
	 * The types of the beans already loaded.
	 */
	private static Map<String, Class> beanTypes = new HashMap<String, Class>();

	/**
	 * Logger for this class.
	 */
	private static Logger log = Logger.getLogger(BeanLocatorUtil.class
			.getName());

	/**
	 * Adds a new bean locator.
	 * 
	 * @param beanLocator
	 *            the bean locator to be added
	 */
	public static void addBeanLocator(BeanLocator beanLocator) {

		if (beanLocators.containsKey(beanLocator.getBeanLocatorId())) {
			removeBeanLocator(beanLocator.getBeanLocatorId());
			log.warning("There is already a beanLocator with the name "
					+ beanLocator.getBeanLocatorId()
					+ ". Deleting previous instance.");
		}
		beanLocators.put(beanLocator.getBeanLocatorId(), new BeanLocatorEntry(
				beanLocator));

	}

	/**
	 * Returns true if at least one of the bean locators are able to locate the
	 * bean specified with this name.
	 * 
	 * @param name
	 *            the name of the bean
	 * @return true if the bean can be found, false otherwise
	 */
	public static boolean containsBean(String name) {
		if (null != getBean(name)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Returns the type of the specified bean.
	 * 
	 * @param name
	 *            the name of the bean
	 * @return the {@link java.lang.Class} of the bean
	 */
	public static Class getBeanType(String name) {
		Class beanType = beanTypes.get(name);
		synchronized (BeanLocatorUtil.class) {
			if (beanType == null) {
				Iterator<BeanLocatorEntry> iter = beanLocators.values()
						.iterator();
				while (beanType == null && iter.hasNext()) {
					BeanLocatorEntry ble = iter.next();
					beanType = ble.beanLocator.getBeanType(name);
					if (beanType != null) {
						beanTypes.put(name, beanType);
					}
				}
			}
		}
		return beanType;
	}

	/**
	 * Returns a bean by name.
	 * 
	 * @param name
	 *            the bean name
	 * @return the bean
	 */
	public static Object getBean(String name) {
		Object entry = beans.get(name);
		synchronized (BeanLocatorUtil.class) {
			if (entry == null) {
				Iterator<BeanLocatorEntry> iter = beanLocators.values()
						.iterator();
				while (entry == null && iter.hasNext()) {
					BeanLocatorEntry ble = iter.next();
					entry = ble.beanLocator.getBean(name);
					if (entry != null) {
						entry = wrapObject(entry);
						beans.put(name, entry);
						ble.containedBeans.add(name);
					}
				}
			}
		}
		return entry;
	}

	/**
	 * Removes the given bean locator from the bean locators.
	 * 
	 * @param beanLocator
	 *            the bean locator to be unregistered
	 */
	public static void removeBeanLocator(BeanLocator beanLocator) {
		removeBeanLocator(beanLocator.getBeanLocatorId());
	}

	/**
	 * Removes the bean locator with the given id.
	 * 
	 * @param beanLocatorId
	 *            the ID of the bean locator to be removed
	 */
	public static void removeBeanLocator(String beanLocatorId) {
		synchronized (BeanLocatorUtil.class) {
			BeanLocatorEntry ble = beanLocators.get(beanLocatorId);
			if (ble != null) {
				Set<String> containedBeans = ble.containedBeans;
				for (String name : containedBeans) {
					beans.remove(name);
				}
				beanLocators.remove(beanLocatorId);
			}
		}
	}

	/**
	 * Creates a {java.lang.reflect.Proxy} object to the given object based on
	 * their interfaces.
	 * 
	 * @param obj
	 *            the object to be proxied
	 * @return the proxy object
	 * @see java.lang.reflect.Proxy
	 */
	private static Object wrapObject(Object obj) {
		return (Object) Proxy.newProxyInstance(obj.getClass().getClassLoader(),
				obj.getClass().getInterfaces(),
				new ClassLoaderRestoreInvocationHandler(obj));
	}

}
